package com.dr.util;

import org.apache.commons.lang3.StringUtils;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;


/**
 * 说明:日期格式转换工具类
 */
public class DateUtil {


    public static final String patternYM = "yyyy-MM";
    /**
     * @param yyyy
     * -MM-dd
     * @return
     */
    public static final String patternA = "yyyy-MM-dd";
    /**
     * @param yyyyMMdd
     * @return
     */
    public static final String patternB = "yyyyMMdd";
    /**
     * @param yyyy-MM-dd HH-mm-ss
     * @return
     */
    public static final String patternC = "yyyy-MM-dd HH-mm-ss";
    /**
     * @param yyyy-MM-dd HH:mm:ss
     * @return
     */
    public static final String patternD = "yyyy-MM-dd HH:mm:ss";
    /**
     * @param yyyy-MM-dd HH:mm
     * @return
     */
    public static final String patternE = "yyyy-MM-dd HH:mm";
    /**
     * @param yyyyMMddHHmmss
     * @return
     */
    public static final String patternF = "yyyyMMddHHmmss";
    /**
     * @param yyyy/MM/dd HH:mm
     * @return
     */
    public static final String patternG = "yy/MM/dd HH:mm";

    private static DateUtil instance;

    public static DateUtil getInstance() {
        if (instance == null) {
            instance = new DateUtil();
        }
        return instance;
    }

    /**
     * (日期转字符串)
     *
     * @param date
     * @param format
     * @return
     * @Title date2String
     */
    public static String dateString(Date date, String format) {
        SimpleDateFormat dateformat = new SimpleDateFormat(format);
        return dateformat.format(date);
    }

    /**
     * 格式化日期为yyyy-MM-dd
     *
     * @param date
     * @return
     */
    public static String formateDate(Date date) {
        return dateToString(date, patternA);
    }

    /**
     * @param //取当天日期
     * @return
     */
    public Date getDate() {
        return Calendar.getInstance().getTime();
    }

    /**
     * @param //取指定年月日的日期 ,格式为yyyy-MM-dd,HH-mm-ss 00-00-00
     * @return
     */
    public static Date getDate(int year, int month, int day) {
        Calendar cal = Calendar.getInstance();
        cal.set(year, month - 1, day, 0, 0, 0);
        return cal.getTime();

    }

    /**
     * @param //取指定年月日时分秒的时间
     * @return
     */
    public Date getDate(int year, int month, int date, int hour, int mintue,
                        int second) {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.YEAR, year);
        cal.set(Calendar.MONTH, month - 1);
        cal.set(Calendar.DATE, date);
        cal.set(Calendar.HOUR_OF_DAY, hour);
        cal.set(Calendar.MINUTE, mintue);
        cal.set(Calendar.SECOND, second);
        return cal.getTime();
    }

    /**
     * @param days=n?,则取n天前,n,则取n天后的日期
     * @param date
     * @param days
     * @return
     */
    public static Date getSomeDaysBeforeAfter(Date date, int days) {
        GregorianCalendar gc = new GregorianCalendar();
        gc.setTime(date);
        gc.add(5, days);
        gc.set(gc.get(Calendar.YEAR), gc.get(Calendar.MONTH),
                gc.get(Calendar.DATE));
        return gc.getTime();
    }

    /**
     * 获取变更后的时间
     *
     * @param date   原时间
     * @param type   变动的单位
     * @param months 数量
     * @return
     */
    public static Date getSomeTimeBeforeAfter(Date date, int type, int months) {
        GregorianCalendar gc = new GregorianCalendar();
        gc.setTime(date);
        gc.add(type, months);
        gc.set(gc.get(Calendar.YEAR), gc.get(Calendar.MONTH), gc.get(Calendar.DATE));
        return gc.getTime();
    }

    /**
     * @param //取指定日期年份
     * @return
     */
    public static int getDateYear(Date date) {

        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.get(Calendar.YEAR);
    }

    /**
     * @param //取指定日期月份
     * @return
     */
    public static int getDateMonth(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.get(Calendar.MONTH) + 1;
    }

    /**
     * @param //取指定日期日
     * @return
     */
    public static int getDateDay(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.get(Calendar.DATE);
    }

    /**
     * @param //取指定日期小时
     * @return
     */
    public int getDateHour(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.get(Calendar.HOUR_OF_DAY);
    }

    /**
     * @param //取指定日期分钟
     * @return
     */
    public int getDateMinute(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.get(Calendar.MINUTE);
    }

    /**
     * @param //取指定日期的第二天的时间小时秒为00:00:00
     * @return
     */
    public Date getNextDayStartTime(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return getNextDayStart(c.get(Calendar.YEAR), c.get(Calendar.MONTH) + 1,
                c.get(Calendar.DATE));
    }

    /**
     * @param //取指定年日的下一日的时间,小时,秒为00:00:00
     * @param //主要是用来取跨月份的日期
     * @return
     */
    public Date getNextDayStart(int year, int month, int date) {
        month = month - 1;
        boolean lastDayOfMonth = false;
        boolean lastDayOfYear = false;

        Calendar time = Calendar.getInstance();
        time.set(year, month, date, 0, 0, 0);
        Calendar nextMonthFirstDay = Calendar.getInstance();
        nextMonthFirstDay.set(year, month + 1, 1, 0, 0, 0);

        if (time.get(Calendar.DAY_OF_YEAR) + 1 == nextMonthFirstDay
                .get(Calendar.DAY_OF_YEAR))
            lastDayOfMonth = true;

        if (time.get(Calendar.DAY_OF_YEAR) == time.getMaximum(Calendar.DATE))
            lastDayOfYear = true;

        time.roll(Calendar.DATE, 1);

        if (lastDayOfMonth)
            time.roll(Calendar.MONTH, 1);

        if (lastDayOfYear)
            time.roll(Calendar.YEAR, 1);

        return time.getTime();
    }


    /**
     * @Date: 2020/3/16 11:15
     * @Author: yangxin
     * @Description: TODO 取指定日期的下一日的时间
     */
    public static Date nextDate(Date date, int amount) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DATE, amount);
        return cal.getTime();
    }

    /**
     * @Date: 2020/3/16 11:15
     * @Author: yangxin
     * @Description: TODO 取指定日期的下一日的时间字符串
     */
    public static String nextDateStr(Date date, int amount) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DATE, amount);
        return dateToString(cal.getTime(), patternD);
    }

    /**
     * @param //指定日期的下的开始时 ,小时,秒为00:00:00
     * @return
     */
    public Date getStartDateNext(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DATE, 1);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        return cal.getTime();
    }

    /**
     * 获取当天开始时间
     * @return
     */
    public static Date getDayBegin(){
        Calendar cal=Calendar.getInstance();
        cal.set(Calendar.HOUR_OF_DAY, 0);//0点
        cal.set(Calendar.MINUTE, 0);//0分
        cal.set(Calendar.SECOND, 0);//0秒
        cal.set(Calendar.MILLISECOND, 0);//0毫秒
        return cal.getTime();
    }

    /**
     * @param //指定日期的开始时 ,小时,秒为00:00:00
     * @return
     */
    public Date getStartDateDay(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        return cal.getTime();
    }

    /**
     * 获取某天前的开始时间
     * @return
     */
    public static Date getBeginDayOfYesterday(Integer num){
        Calendar cal=Calendar.getInstance();
        cal.setTime(getDayBegin());//当天开始时间
        cal.add(Calendar.DAY_OF_MONTH, -num);//当天月份天数减1
        return cal.getTime();
    }

    /**
     * @param //指定日期的结束时 ,小时,秒为23:59:59
     * @return
     */
    public Date getEndDateDay(Date date) {
        if (date == null)
            return null;
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.HOUR_OF_DAY, 23);
        cal.set(Calendar.MINUTE, 59);
        cal.set(Calendar.SECOND, 59);
        return cal.getTime();
    }

    /**
     * @param //将指定日 ,以指定pattern格式,输出String
     * @return
     */
    public static String dateToString(Date date, String pattern) {
        if (date == null) {
            return "";
        } else {
            SimpleDateFormat format = new SimpleDateFormat(pattern);
            return format.format(date);
        }
    }



    /**
     * @param //将指定字符型日期转为日期 ,,格式为指定的pattern
     * @return
     */
    public static Date stringToDate(String string, String pattern) {
        SimpleDateFormat format = (SimpleDateFormat) DateFormat
                .getDateInstance();
        format.applyPattern(pattern);
        try {
            return format.parse(string);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * @param //将指定字符型日期转为日期 ,指定格式为yyyy-MM-dd
     * @return
     */
    public static Date stringToDate(String string) {
        return stringToDate(string, patternA);
    }

    /**
     * 获得两个日期之间间隔的天
     *
     * @param startDate 起始年月
     * @param endDate   结束年月
     * @return int
     */
    public static int getDays(Date startDate, Date endDate) {
        int elapsed = 0;
        Calendar cal = Calendar.getInstance();
        cal.setTime(startDate);
        Date d1 = cal.getTime();

        cal.setTime(endDate);
        Date d2 = cal.getTime();

        long daterange = d2.getTime() - d1.getTime();
        long time = 1000 * 3600 * 24; // �?��的毫秒数
        elapsed = (int) (daterange / time);
        return elapsed;
    }

    public static long getDay(String start, String end) throws ParseException {
        SimpleDateFormat sf = new SimpleDateFormat("yyyy-MM-dd");
        Long c = sf.parse(end).getTime() - sf.parse(start).getTime();
        long d = c / 1000 / 60 / 60 / 24;//天
        return d;
    }

    /**
     * @param date
     * @param startTime 格800，表示上
     * @param //endTime格式
     * @return
     */
    public boolean isWorkHour(Date date, String startTime, String endTime) {// 是否是工作时�?
        if (StringUtils.isEmpty(startTime))
            startTime = "0800";
        if (StringUtils.isEmpty(endTime))
            endTime = "2200";
        int start = Integer.parseInt(startTime);
        int end = Integer.parseInt(endTime);
        int hour = getDateHour(date);
        int m = getDateMinute(date);
        String hstr = hour <= 9 ? "0" + hour : hour + "";
        String mstr = m <= 9 ? "0" + m : m + "";
        int dateInt = Integer.parseInt(hstr + mstr);
        if (dateInt >= start && dateInt <= end) {
            return true;
        }
        return false;
    }

    // 取日期的当前月
    public static Date getMonthFirstDay(Date date) {
        return getDate(getDateYear(date), getDateMonth(date), 1);
    }

    /**
     * 获取当前时间字符
     *
     * @return
     */
    public static String getDateOfNow() {
        return dateToString(new Date(), patternC);
    }

    /**
     * 将时间戳转换成时间字符串
     *
     * @param stamp php的时间戳
     * @return yyyy-MM-dd HH:mm:ss
     */
    public static String changeTimeStampToString(long stamp) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String dateTime = null;
        try {
            if (stamp < 9999999999L)
                stamp = stamp * 1000L;
            dateTime = df.format(stamp);
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return dateTime;
    }

    /**
     * 将时间戳转换成时间字符串
     *
     * @param stamp   php的时间戳
     * @param pattern 转换的样
     * @return 转换后的结果, 仪pattern为准
     */
    public static String changeTimeStampToString(long stamp, String pattern) {
        SimpleDateFormat df = new SimpleDateFormat(pattern);
        String dateTime = null;
        try {
            if (stamp < 9999999999L)
                stamp = stamp * 1000L;
            dateTime = df.format(stamp);
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return dateTime;
    }

    /**
     * (字符串转日期)
     *
     * @param date
     * @param format
     * @return
     * @Title string2Date
     */
    public static Date string2Date(String date,
                                   String format) {

        SimpleDateFormat dateformat = new SimpleDateFormat(format);
        try {
            return dateformat.parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * (日期转字符串)
     *
     * @param date
     * @param format
     * @return
     * @Title date2String
     */
    public static String date2String(Date date,
                                     String format) {
        SimpleDateFormat dateformat = new SimpleDateFormat(format);
        return dateformat.format(date);
    }


    /**
     * 日期转字符串:yyyy-MM-dd HH:mm
     *
     * @return
     * @Title date2String
     */
    public static String dateToStr() {
        return dateToString(new Date(), patternC);
    }

    /**
     * 获得指定日期的后
     *
     * @param specifiedDay
     * @return
     */
    public static String getSpecifiedDayAfter(String specifiedDay) {
        Calendar c = Calendar.getInstance();
        Date date = null;
        try {
            date = new SimpleDateFormat("yy-MM-dd").parse(specifiedDay);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        c.setTime(date);
        int day = c.get(Calendar.DATE);
        c.set(Calendar.DATE, day + 1);

        String dayAfter = new SimpleDateFormat("yyyy-MM-dd").format(c.getTime());
        return dayAfter;
    }

    /**
     * 获得指定日期前几天的日期
     *
     * @param specifiedDay
     * @return
     */
    public static String getSpecifiedDay(String specifiedDay, int days) {
        Calendar c = Calendar.getInstance();
        Date date = null;
        try {
            date = new SimpleDateFormat("yy-MM-dd").parse(specifiedDay);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        c.setTime(date);
        int day = c.get(Calendar.DATE);
        c.set(Calendar.DATE, day - days);

        String dayAfter = new SimpleDateFormat("yyyy-MM-dd").format(c.getTime());
        return dayAfter;
    }

    /**
     * 日期间相差的秒数
     *
     * @param startDate
     * @return
     */
    public static int calLastedTime(Date startDate, Date endDate) {
        long a = endDate.getTime();
        long b = startDate.getTime();
        int c = (int) ((a - b) / 1000);
        return c;
    }

    /**
     * 获取上个月年月
     *
     * @return
     */
    public static String getPreYearMonth(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        calendar.add(Calendar.MONTH, -1);
        date = calendar.getTime();
        return new SimpleDateFormat("yyyy-MM").format(date);
    }

    /**
     * 获取当前月的第一天
     */
    public static Date getSatrtMonthDay() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, 0);
        c.set(Calendar.DAY_OF_MONTH, 1);
        return c.getTime();
    }

    /**
     * 获取下个月年月
     *
     * @return
     */
    public static String getNextYearMonth(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        calendar.add(Calendar.MONTH, 1);
        date = calendar.getTime();
        return new SimpleDateFormat("yyyy-MM").format(date);
    }

    public static String getAfterSomeMonth(Date date, int months) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        calendar.add(Calendar.MONTH, months);
        date = calendar.getTime();
        return new SimpleDateFormat("yyyy-MM").format(date);
    }

    /**
     * 获取日期的次月5号,写成6号便于判断日期
     *
     * @return
     */
    public static String getNextDate(Date date) {
        date.setDate(6);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, 1);
        date = calendar.getTime();
        return new SimpleDateFormat("yyyy-MM-dd").format(date);
    }

    /**
     * 判断俩个日期是否在同一个月
     *
     * @param date1
     * @param date2
     * @return
     */
    public static boolean isSameYM(Date date1, Date date2) {
        boolean istrue = false;
        Calendar c1 = Calendar.getInstance();
        Calendar c2 = Calendar.getInstance();
        c1.setTime(date1);
        c2.setTime(date2);
        if (c1.get(Calendar.YEAR) == c2.get(Calendar.YEAR) && c1.get(Calendar.MONTH) == c2.get(Calendar.MONTH))
            istrue = true;
        return istrue;
    }

    public static boolean isSameDay(Date date1, Date date2) {
        return DateUtil.stringToDate(DateUtil.dateToString(date1, patternA), DateUtil.patternA).compareTo(DateUtil.stringToDate(DateUtil.dateToString(date2, patternA), patternA)) == 0;
    }

    public static Date getLastDay(Date date) {
        Calendar ca = Calendar.getInstance();
        ca.setTime(date);
        ca.set(Calendar.DATE, 1);
        ca.roll(Calendar.DATE, -1);
        return ca.getTime();
    }

    /**
     * 根据用户生日计算年龄
     *
     * @author CJJ
     */
    public static int getAgeByBirthday(Date birthday) {
        Calendar cal = Calendar.getInstance();

        if (cal.before(birthday)) {
            throw new IllegalArgumentException(
                    "The birthDay is before Now.It's unbelievable!");
        }

        int yearNow = cal.get(Calendar.YEAR);
        int monthNow = cal.get(Calendar.MONTH) + 1;
        int dayOfMonthNow = cal.get(Calendar.DAY_OF_MONTH);

        cal.setTime(birthday);
        int yearBirth = cal.get(Calendar.YEAR);
        int monthBirth = cal.get(Calendar.MONTH) + 1;
        int dayOfMonthBirth = cal.get(Calendar.DAY_OF_MONTH);

        int age = yearNow - yearBirth;

        if (monthNow <= monthBirth) {
            if (monthNow == monthBirth) {
                if (dayOfMonthNow < dayOfMonthBirth) {
                    age--;
                }
            } else {
                age--;
            }
        }
        return age;
    }

    /**
     * Date转时间戳
     *
     * @param data
     * @return
     * @author zx
     * @date 2018年2月26日 下午5:14:54
     */
    public static Integer dateForStamp(Date data) {
        return (int) (data.getTime() / 1000);
    }

    /**
     * 获取当天任何时间点的时间戳 例如 ：getDateString（23,59,59）yyyy-23-59-59
     * @param minute
     * @param second
     * @return
     */
    public static String getDateString(Integer hr, Integer minute, Integer second) {
        Calendar calendar = Calendar.getInstance();
        if (hr == null) {
            calendar.set(Calendar.HOUR_OF_DAY, 0);
        }else {
            if (hr != 0 ) {
                calendar.set(Calendar.HOUR_OF_DAY, hr);
            }
        }
        if (minute == null) {
            calendar.set(Calendar.MINUTE, 0);
        } else {
            calendar.set(Calendar.MINUTE, minute);
        }
        if (second == null) {
            calendar.set(Calendar.SECOND, 0);
        } else {
            calendar.set(Calendar.SECOND, second);

        }
        return String.valueOf(calendar.getTimeInMillis());
    }

    /**
     * 获取当天零点时间戳
     * @return
     */
    public static String getDateString() {
        return getDateString(null, null, null);
    }

    /**
     * 获取当前小时下任何时间点的时间戳
     * @param minute
     * @param second
     * @return
     */
    public static String getDateString(Integer minute, Integer second) {
        return getDateString(0, minute,second);
    }


    /**
     * 获取当前小时下零分零秒的时间戳
     * @return
     */
    public static String getDateHrString() {
        return getDateString(0, null,null);
    }

    /**
     * @Date: 2019/8/7 16:48
     * @Author: yangxin
     * @Description: TODO 获取周期内的起始时间
     * @param period 周期
     * @param starting_time 生效时间
     */
    public static Date getTimeRange(Integer period ,Date starting_time){
        Date today = new Date();
        //获取到两个时间的时间差，当前时间 减去 生效时间 等于 时间差（days）
        long diff = today.getTime() - starting_time.getTime();
        long days = diff / (1000 * 60 * 60 * 24);
        //将时间差 取余，得知当前时间 在周期中已经 过去了第几天， 我们将这个值名为时间节点（timeNode）
        Integer timeNode = (int)days % period;
        //调用 获取某天前的开始时间 方法 得到 计算后的最新生效时间
        return getBeginDayOfYesterday(timeNode);
    }

    /**
     * 传入时间与当前时间的时间差
     * @param date
     * @return
     */
    public static int timeDifference(Date date)
    {
        int days = (int) ((date.getTime() - new Date().getTime()) / (1000*3600*24));
        return days+1;
    }



    /**
     *
     * @param date1 <String>
     * @param date2 <String>
     * @return int
     * @throws ParseException  两个日期相差月数
     */
    public static Long getMonthSpace(String date1, String date2){
        long result = 0;
        Calendar c1 = Calendar.getInstance();
        c1.setTime(stringToDate(date1));
        Calendar c2 = Calendar.getInstance();
        c2.setTime(stringToDate(date2));
        //只要年月
        long fromYear = c1.get(Calendar.YEAR);
        long fromMonth = c1.get(Calendar.MONTH);
        long toYear = c2.get(Calendar.YEAR);
        long toMonth = c2.get(Calendar.MONTH);
        result = toYear *  12  + toMonth  -  (fromYear  *  12  +  fromMonth);
        return result == 0 ? 1 : Math.abs(result);
    }

    /**
     * 获取前多少天的时间---一天的最小或最大的时间
     *
     * @param days         前多少天
     * @param format       格式化
     * @param minOrMaxDate 最小（0）或最大（1）
     * @return string 最终的时间
     */
    public static String getBeforeTime(long days, String format, Integer minOrMaxDate) {
        if (minOrMaxDate.equals(0)) {
            return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().with(LocalTime.MIN).plusDays(-days));
        }
        return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().with(LocalTime.MAX).plusDays(-days));
    }

    /**
     * LocalDateTime转Date，格式为yyyy-MM-dd
     *
     * @param  now 时间
     * @return Date
     */
    /*
    public static Date getDateByYyyyMmDd(LocalDateTime now){
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(DateUtil.patternA);
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(DateUtil.patternA);
        Date date;
        try {
            date = simpleDateFormat.parse(dateTimeFormatter.format(now));
        } catch (ParseException pe) {
            throw new BusinessRuntimeException(BusinessCode.C_4000, "日期格式化错误");
        }
        return date;
    }
     */

    /**
     * String格式转换
     *
     * @param time       日期
     * @param pattern    转换前格式
     * @param resPattern 转换后格式
     * @return 时间
     */
    public static String getStringByYyMmDd(String time, String pattern, String resPattern) {
        DateTimeFormatter inputFormatter = DateTimeFormatter.ofPattern(pattern);
        LocalDateTime dateTime = LocalDateTime.parse(time, inputFormatter);
        DateTimeFormatter outputFormatter = DateTimeFormatter.ofPattern(resPattern);
        return dateTime.format(outputFormatter);
    }

    /**
     * 获取前一个季度 以1-3月开始
     *
     * @return 季度
     */
    public static String getPreviousQuarter() {
        LocalDate currentDate = LocalDate.now();
        // 获取当前月份
        Month currentMonth = currentDate.getMonth();
        // 计算前一个季度的月份范围
        int startMonth;
        int endMonth;
        if (currentMonth == Month.JANUARY || currentMonth == Month.FEBRUARY || currentMonth == Month.MARCH) {
            // 当前是1-3月，前一个季度是10-12月
            startMonth = 10;
            endMonth = 12;
        } else if (currentMonth == Month.APRIL || currentMonth == Month.MAY || currentMonth == Month.JUNE) {
            // 当前是4-6月，前一个季度是1-3月
            startMonth = 1;
            endMonth = 3;
        } else if (currentMonth == Month.JULY || currentMonth == Month.AUGUST || currentMonth == Month.SEPTEMBER) {
            // 当前是7-9月，前一个季度是4-6月
            startMonth = 4;
            endMonth = 6;
        } else {
            // 当前是10-12月，前一个季度是7-9月
            startMonth = 7;
            endMonth = 9;
        }
        return startMonth + "-" + endMonth;
    }


    /**
     * 获取季度 年月
     *
     * @return
     */
    public static String getYearPreviousQuarter() {
        LocalDate currentDate = LocalDate.now();
        // 获取当前月份
        Month currentMonth = currentDate.getMonth();
        // 获取当前年份
        int currentYear = currentDate.getYear();
        // 计算前一个季度的月份范围
        int startMonth;
        int endMonth;
        if (currentMonth == Month.JANUARY || currentMonth == Month.FEBRUARY || currentMonth == Month.MARCH) {
            // 当前是1-3月，前一个季度是10-12月
            startMonth = 10;
            endMonth = 12;
            currentYear--;
        } else if (currentMonth == Month.APRIL || currentMonth == Month.MAY || currentMonth == Month.JUNE) {
            // 当前是4-6月，前一个季度是1-3月
            startMonth = 1;
            endMonth = 3;
        } else if (currentMonth == Month.JULY || currentMonth == Month.AUGUST || currentMonth == Month.SEPTEMBER) {
            // 当前是7-9月，前一个季度是4-6月
            startMonth = 4;
            endMonth = 6;
        } else {
            // 当前是10-12月，前一个季度是7-9月
            startMonth = 7;
            endMonth = 9;
        }
        return currentYear + "-" + startMonth + "-" + currentYear + "-" + endMonth;
    }

    /**
     * 获取上个月的最小最大时间
     *
     * @return 最小 最大时间
     */
    public static long[] getLastMonthMinMaxTimestamp() {
        // 获取当前日期和时间
        LocalDateTime currentDateTime = LocalDateTime.now();

        // 获取上个月的日期
        LocalDateTime lastMonthDateTime = currentDateTime.minusMonths(1);

        // 设置日期为上个月的第一天，并设置时间为最小时间（00:00:00）
        LocalDateTime firstDayOfMonth = LocalDateTime.of(lastMonthDateTime.getYear(), lastMonthDateTime.getMonth(), 1, 0, 0, 0);

        // 设置日期为上个月的最后一天，并设置时间为最大时间（23:59:59）
        LocalDateTime lastDayOfMonth = LocalDateTime.of(lastMonthDateTime.getYear(), lastMonthDateTime.getMonth(), lastMonthDateTime.getMonth().maxLength(), 23, 59, 59);

        // 转换为时间戳
        long minTimestamp = firstDayOfMonth.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
        long maxTimestamp = lastDayOfMonth.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();

        return new long[]{minTimestamp, maxTimestamp};
    }

    /**
     * localDateTime 转 string
     *
     * @param localDateTime 时间
     * @param format 格式
     * @return 时间
     */
    public static String localDateTimeToString(LocalDateTime localDateTime, String format) {

        return localDateTime.format(DateTimeFormatter.ofPattern(format));
    }

    /**
     * string 转 long
     *
     * @param time 时间
     * @param format 格式
     * @return 时间
     */
    /*
    public static long stringToTimeStamp(String time, String format){
        SimpleDateFormat dateFormat = new SimpleDateFormat(format);
        try {
            Date date = dateFormat.parse(time);
            return date.getTime();
        } catch (ParseException pe) {
            throw new BusinessRuntimeException(BusinessCode.C_4000, "日期格式化错误");
        }
    }
     */

    /**
     * long 转 string
     *
     * @param time 时间
     * @param format 格式
     * @return 时间
     */
    public static String timeStampToString(long time, String format) {
        return DateTimeFormatter.ofPattern(format)
                .withZone(ZoneId.systemDefault())
                .format(Instant.ofEpochMilli(time));
    }

    /**
     * long 转 localdatetime
     *
     * @param time 时间
     * @return 时间
     */
    public static LocalDateTime longToTimeStamp(long time) {

        return Instant.ofEpochMilli(time).atZone(ZoneId.systemDefault()).toLocalDateTime();
    }

    /**
     * localdatetime 转 long
     *
     * @param localDateTime 时间
     * @return 时间
     */
    public static long timeStampToLong(LocalDateTime localDateTime) {

        return localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    /**
     * 获取去年处于当前季度的最小最大时间 3-5 6-8...
     *
     * @return 时间数组 yyyy-MM-dd HH:mm:ss
     */
    public static String[] getCurrentSeasonTimeRange() {
        LocalDateTime currentDateTime = LocalDateTime.now();
        int year = currentDateTime.plusYears(-1).getYear();
        Month currentMonth = currentDateTime.getMonth();
        String currentSeason = getSeason(currentMonth);
        LocalDateTime seasonStart = getSeasonStart(year, currentSeason);
        LocalDateTime seasonEnd = getSeasonEnd(year, currentSeason);
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(patternA);
        return new String[]{dateTimeFormatter.format(seasonStart), dateTimeFormatter.format(seasonEnd)};
    }

    public static String getSeason(Month month) {
        if (month == Month.MARCH || month == Month.APRIL || month == Month.MAY) {
            return "Spring";
        } else if (month == Month.JUNE || month == Month.JULY || month == Month.AUGUST) {
            return "Summer";
        } else if (month == Month.SEPTEMBER || month == Month.OCTOBER || month == Month.NOVEMBER) {
            return "Autumn";
        } else {
            return "Winter";
        }
    }

    public static LocalDateTime getSeasonStart(int year, String season) {
        if ("Spring".equals(season)) {
            return LocalDateTime.of(year, Month.MARCH, 1, 0, 0, 0);
        } else if ("Summer".equals(season)) {
            return LocalDateTime.of(year, Month.JUNE, 1, 0, 0, 0);
        } else if ("Autumn".equals(season)) {
            return LocalDateTime.of(year, Month.SEPTEMBER, 1, 0, 0, 0);
        } else {
            return LocalDateTime.of(year, Month.DECEMBER, 1, 0, 0, 0);
        }
    }

    public static LocalDateTime getSeasonEnd(int year, String season) {
        if ("Spring".equals(season)) {
            return LocalDateTime.of(year, Month.MAY, 31, 23, 59, 59);
        } else if ("Summer".equals(season)) {
            return LocalDateTime.of(year, Month.AUGUST, 31, 23, 59, 59);
        } else if ("Autumn".equals(season)) {
            return LocalDateTime.of(year, Month.NOVEMBER, 30, 23, 59, 59);
        } else {
            year = year + 1;
            YearMonth february = YearMonth.of(year, Month.FEBRUARY);
            int lastDayOfMonth = february.lengthOfMonth();
            return LocalDateTime.of(year, Month.FEBRUARY, lastDayOfMonth, 23, 59, 59);
        }
    }

    /**
     * 获取上个季度的时间
     *
     * @return LocalDateTime[]
     */
    public static LocalDateTime[] getLastQuarterDates() {
        LocalDateTime[] dates = new LocalDateTime[2];
        LocalDate currentDate = LocalDate.now();
        int currentMonthValue = currentDate.getMonthValue();
        int firstMonthValue;
        int lastMonthValue;
        if (currentMonthValue <= 3) {
            // 10月
            firstMonthValue = 10;
            // 12月
            lastMonthValue = 12;
        } else if (currentMonthValue <= 6) {
            // 1月
            firstMonthValue = 1;
            // 3月
            lastMonthValue = 3;
        } else if (currentMonthValue <= 9) {
            // 4月
            firstMonthValue = 4;
            // 6月
            lastMonthValue = 6;
        } else {
            // 7月
            firstMonthValue = 7;
            // 9月
            lastMonthValue = 9;
        }
        LocalDate firstDayOfLastQuarter = LocalDate.of(currentDate.getYear(), firstMonthValue, 1);
        LocalDate lastDayOfLastQuarter = LocalDate.of(currentDate.getYear(), lastMonthValue, 1)
                .with(TemporalAdjusters.lastDayOfMonth());
        LocalDateTime minDate = firstDayOfLastQuarter.atStartOfDay();
        LocalDateTime maxDate = lastDayOfLastQuarter.atTime(23, 59, 59);
        dates[0] = minDate;
        dates[1] = maxDate;
        return dates;
    }

}