package com.dr.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class NewDateUtil {

    public static final String PATTERN_A = "yyyy-MM-dd";
    public static final String PATTERN_B = "yyyy-MM-dd HH:mm:ss";

    public static final String PATTERN_C = "yyyyMMdd";

    public static final String PATTERN_D = "yyyy";

    /**
     * 日期转字符串
     *
     * @param date    日期
     * @param pattern 指定日期格式
     */
    public static String toStr(Date date, String pattern) {
        if (date == null) return "";
        SimpleDateFormat format = new SimpleDateFormat(pattern);
        return format.format(date);
    }

    /**
     * 将日期字符串转为日期
     *
     * @param string  字符型日期
     * @param pattern 指定格式
     */
    public static Date toDate(String string, String pattern) {
        SimpleDateFormat format = (SimpleDateFormat) DateFormat.getDateInstance();
        format.applyPattern(pattern);
        try {
            return format.parse(string);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * 自定义加减多少个小时
     * 传入具体日期 ，返回具体日期加减一个小时。
     *
     * @param date      日期(2014-04-20 01:01:01)
     * @param hourCount 小时 (+1,-1)
     * @return 2014-04-20 02:01:01
     * @throws ParseException
     */
    public static String editHour(String date, Integer hourCount) throws ParseException {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Date dt = sdf.parse(date);
        Calendar rightNow = Calendar.getInstance();
        rightNow.setTime(dt);

        rightNow.add(Calendar.HOUR_OF_DAY, hourCount);
        Date dt1 = rightNow.getTime();
        return sdf.format(dt1);
    }


    /**
     * 自定义加减多少秒
     * 传入具体日期 ，返回具体日期加减一个小时。
     *
     * @param date    当前日期(2014-04-20 01:01:01)
     * @param seconds 秒数 (+10,-10)
     * @return 2014-04-20 01:01:11
     * @throws ParseException
     */
    public static Date addSeconds(Date date, int seconds) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.SECOND, seconds);
        return calendar.getTime();
    }

    /**
     * 校验日期格式 yyyy-MM-dd
     */
    public static boolean inspectDateFormat(String dateSt) {
        try {
            String regEx = "^((?!0000)[0-9]{4}-((0[1-9]|1[0-2])-(0[1-9]|1[0-9]|2[0-8])|(0[13-9]|1[0-2])-(29|30)|(0[13578]|1[02])-31)|([0-9]{2}(0[48]|[2468][048]|[13579][26])|(0[48]|[2468][048]|[13579][26])00)-02-29)$";
            Matcher matcher = Pattern.compile(regEx).matcher(dateSt);
            if (matcher.find()) {
                return true;
            }
            return false;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 获得两个日期之间间隔的天
     *
     * @param startDate 起始年月
     * @param endDate   结束年月
     * @return int
     */
    public static int getDays(Date startDate, Date endDate) {
        int elapsed = 0;
        Calendar cal = Calendar.getInstance();
        cal.setTime(startDate);
        Date d1 = cal.getTime();

        cal.setTime(endDate);
        Date d2 = cal.getTime();

        long daterange = d2.getTime() - d1.getTime();
        long time = 1000 * 3600 * 24; // �?��的毫秒数
        elapsed = (int) (daterange / time);
        return elapsed;
    }

    /**
     * 获得指定日期前几天的日期
     *
     * @return
     */
    public static Date getSpecifiedDay(int days) {
        Calendar calen = Calendar.getInstance();
        calen.setTime(new Date());
        calen.set(Calendar.DATE, calen.get(Calendar.DATE) - days);
        return calen.getTime();
    }

    /**
     * 当天的起始时间
     *
     * @return
     */
    public static String todayStartTime() {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        SimpleDateFormat sdf = new SimpleDateFormat(PATTERN_B);
        return sdf.format(calendar.getTime());
    }

    /**
     * 取当前时间前三分钟的时间
     *
     * @return
     */
    public static String getCurrentTime(Integer minute) {
        Calendar calen = Calendar.getInstance();
        calen.setTime(new Date());
        calen.add(Calendar.MINUTE, -minute);
        SimpleDateFormat sdf = new SimpleDateFormat(PATTERN_B);
        return sdf.format(calen.getTime());
    }

    /**
     * 获取当前日期yyyy-mm-dd
     *
     * @return
     */
    public static String getCurrentDate() {
        SimpleDateFormat sdf = new SimpleDateFormat(PATTERN_A);
        return sdf.format(new Date());
    }

    public static int calLastedTime(Date startDate, Date endDate) {
        long nowDate = endDate.getTime();
        long startDateTime = startDate.getTime();
        int diffSeconds = (int) ((nowDate - startDateTime) / 1000);
        return diffSeconds;
    }

    /**
     * 获取前多少天的时间---实时的时间
     *
     * @param days   前多少天
     * @param format 格式化
     * @return string 最终的时间
     */
    public static String getBeforeTime(long days, String format) {

        return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().plusDays(-days));
    }

    /**
     * 获取前几个月的最后一天的时间---实时的时间
     *
     * @param days   前几个月
     * @param format 格式化
     * @return string 最终的时间
     */
    public static String getMonthLastDaysTime(long days, String format) {

        return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().minusMonths(days).with(TemporalAdjusters.lastDayOfMonth()));
    }

    /**
     * 获取前多少天的时间---一天的最小或最大的时间
     *
     * @param days         前多少天
     * @param format       格式化
     * @param minOrMaxDate 最小（0）或最大（1）
     * @return string 最终的时间
     */
    public static String getBeforeTime(long days, String format, Integer minOrMaxDate) {
        if (minOrMaxDate.equals(0)) {
            return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().with(LocalTime.MIN).plusDays(-days));
        }
        return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().with(LocalTime.MAX).plusDays(-days));
    }

    /**
     * 获取上周第一天最小时间和最后一天最大的时间
     *
     * @param week         前几周
     * @param format       格式化
     * @param firstOrLast  第一天（0）或最后一天（1）
     * @param minOrMaxDate 最小（0）或最大（1）
     */
    public static String getLastWeekTime(long week, String format, Integer firstOrLast, Integer minOrMaxDate) {
        if (minOrMaxDate.equals(0)) {
            return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().minusWeeks(week)
                    .with(DayOfWeek.MONDAY)
                    .with(LocalTime.MIN));
        }
        return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().minusWeeks(week)
                .with(DayOfWeek.SUNDAY)
                .with(LocalTime.MAX));
    }

    /**
     * 获取前几个月的第一天或最后一天的时间---一天的最小或最大的时间
     *
     * @param days         前几个月
     * @param format       格式化
     * @param firstOrLast  第一天（0）或最后一天（1）
     * @param minOrMaxDate 最小（0）或最大（1）
     * @return string 最终的时间
     */
    public static String getMonthLastDaysTime(long days, String format, Integer firstOrLast, Integer minOrMaxDate) {
        if (minOrMaxDate.equals(0)) {
            return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().minusMonths(days)
                    .with(firstOrLast.equals(0) ? TemporalAdjusters.firstDayOfMonth() : TemporalAdjusters.lastDayOfMonth())
                    .with(LocalTime.MIN));
        }
        return DateTimeFormatter.ofPattern(format).format(LocalDateTime.now().minusMonths(days)
                .with(firstOrLast.equals(0) ? TemporalAdjusters.firstDayOfMonth() : TemporalAdjusters.lastDayOfMonth())
                .with(LocalTime.MAX));
    }

    /**
     * 获取去年处于当前季度的最小最大时间 3-5 6-8...
     *
     * @return 时间数组 yyyy-MM-dd HH:mm:ss
     */
    public static String[] getCurrentSeasonTimeRange() {
        LocalDateTime currentDateTime = LocalDateTime.now();
        int year = currentDateTime.plusYears(-1).getYear();
        Month currentMonth = currentDateTime.getMonth();
        String currentSeason = getSeason(currentMonth);
        LocalDateTime seasonStart = getSeasonStart(year, currentSeason);
        LocalDateTime seasonEnd = getSeasonEnd(year, currentSeason);
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(PATTERN_B);
        return new String[]{dateTimeFormatter.format(seasonStart), dateTimeFormatter.format(seasonEnd)};
    }

    public static String getSeason(Month month) {
        if (month == Month.MARCH || month == Month.APRIL || month == Month.MAY) {
            return "Spring";
        } else if (month == Month.JUNE || month == Month.JULY || month == Month.AUGUST) {
            return "Summer";
        } else if (month == Month.SEPTEMBER || month == Month.OCTOBER || month == Month.NOVEMBER) {
            return "Autumn";
        } else {
            return "Winter";
        }
    }

    public static LocalDateTime getSeasonStart(int year, String season) {
        if ("Spring".equals(season)) {
            return LocalDateTime.of(year, Month.MARCH, 1, 0, 0, 0);
        } else if ("Summer".equals(season)) {
            return LocalDateTime.of(year, Month.JUNE, 1, 0, 0, 0);
        } else if ("Autumn".equals(season)) {
            return LocalDateTime.of(year, Month.SEPTEMBER, 1, 0, 0, 0);
        } else {
            return LocalDateTime.of(year, Month.DECEMBER, 1, 0, 0, 0);
        }
    }

    public static LocalDateTime getSeasonEnd(int year, String season) {
        if ("Spring".equals(season)) {
            return LocalDateTime.of(year, Month.MAY, 31, 23, 59, 59);
        } else if ("Summer".equals(season)) {
            return LocalDateTime.of(year, Month.AUGUST, 31, 23, 59, 59);
        } else if ("Autumn".equals(season)) {
            return LocalDateTime.of(year, Month.NOVEMBER, 30, 23, 59, 59);
        } else {
            year = year + 1;
            YearMonth february = YearMonth.of(year, Month.FEBRUARY);
            int lastDayOfMonth = february.lengthOfMonth();
            return LocalDateTime.of(year, Month.FEBRUARY, lastDayOfMonth, 23, 59, 59);
        }
    }

    /**
     * 获取近多少天全部日期
     *
     * @param numberOfDays 前多少天
     * @return 日期集合
     */
    public static List<String> getRecentDays(int numberOfDays) {
        List<String> recentDays = new ArrayList<>();
        LocalDateTime now = LocalDateTime.now();
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(PATTERN_A);
        for (int i = numberOfDays - 1; i >= 0; i--) {
            LocalDateTime date = now.minusDays(i);
            recentDays.add(formatter.format(date));
        }
        return recentDays;
    }

    /**
     * 获取当年每月最小最大时间
     *
     * @param year 年份
     * @return 返回时间，例如 2024-08:2024-12
     */
    public static List<String> getMinAndMaxMonthByYear(int year) {
        // 创建 DateTimeFormatter
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(PATTERN_A);
        // 获取每个月的最小和最大时间，并格式化为 yyyy-MM-dd
        List<String> minMaxTimes = new ArrayList<>();
        for (int month = 1; month <= 12; month++) {
            YearMonth yearMonth = YearMonth.of(year, month);
            // 获取该月的第一个和最后一个时间
            LocalDateTime startOfMonth = yearMonth.atDay(1).atStartOfDay();
            LocalDateTime endOfMonth = yearMonth.atEndOfMonth().atTime(23, 59, 59);
            // 格式化时间
            String formattedStart = startOfMonth.format(formatter);
            String formattedEnd = endOfMonth.format(formatter);
            // 添加到列表
            minMaxTimes.add(formattedStart + ":" + formattedEnd);
        }
        return minMaxTimes;
    }

    /**
     * 获取近多少天全部日期
     *
     * @param time 时间
     * @return LocalDate
     */
    public static LocalDate stringToLocalDate(String time, String format) {
        // 将字符串转换为 LocalDate
        return LocalDate.parse(time, DateTimeFormatter.ofPattern(format));
    }

    /**
     * 获取近多少天全部日期
     *
     * @param startAge 开始年龄
     * @param endAge 结束年龄
     * @return LocalDate[]
     */
    public static LocalDate[] getTimeRangeByAge(int startAge, int endAge) {
        // 计算当前日期
        LocalDate currentDate = LocalDate.now();
        // 计算结束年龄对应的出生日期
        LocalDate endDate = currentDate.minusYears(startAge).minusDays(1);
        // 计算结束年龄对应的出生日期
        LocalDate startDate = currentDate.minusYears(endAge);
        return new LocalDate[]{startDate,endDate};
    }
}
